package io.bidmachine.schema.rtb

import com.github.plokhotnyuk.jsoniter_scala.core.JsonValueCodec
import com.github.plokhotnyuk.jsoniter_scala.macros.{CodecMakerConfig, JsonCodecMaker}
import enumeratum.values.{IntEnum, IntEnumEntry}
import io.bidmachine.schema.adcom.{Context, Placement}
import io.bidmachine.schema.analytics.Metrics
import io.bidmachine.schema.analytics.NumericPriceLevels.DoubleLevel
import io.bidmachine.schema.analytics.PriceLevels.Level
import io.bidmachine.schema.annotations.money
import io.bidmachine.schema.jsoniter.JsoniterEnumInstances
import io.bidmachine.schema.rtb.Request.Ext.SupplyParameters

import java.time.Instant

case class Request(
  id: String,
  @money flr: Double,
  test: Boolean,
  at: AuctionType,
  placement: Placement,
  context: Context,
  ext: Request.Ext
)

object Request {

  sealed abstract class PlacementType(val value: Int) extends IntEnumEntry

  object PlacementType extends IntEnum[PlacementType] with JsoniterEnumInstances {

    case object Display extends PlacementType(1)

    case object Video extends PlacementType(2)

    case object Native extends PlacementType(3)

    val values: IndexedSeq[PlacementType] = findValues
  }

  case class Ext(
    adRequestId: String,
    pt: PlacementType,
    adt: Option[String],                   // Ad Type
    int: Option[String],                   // Integration type
    dc: Option[String],                    // Datacenter ID
    adSpace: Option[String],
    network: Option[Boolean],              // Ad Network flag
    adUnit: Option[String],                // Ad Network Unit ID
    multiFlr: Option[Boolean],
    calloutRound: Option[Int],
    calloutBidRound: Option[Int],
    calloutMaxRound: Option[Int],
    calloutPolicy: Option[String],
    bidPrediction: Option[Int],
    sellerFlrPolicy: Option[String],
    bidderFlrPolicy: Option[String],
    flrInitial: Option[Level],             // sellerBidFloor Level
    flrInitialV2: Option[DoubleLevel],     // numeric sellerBidFloor Level
    flrScaled: Option[Level],              // bidFloor Level
    flrScaledV2: Option[DoubleLevel],      // numeric bidFloor Level
    @money flr: Double,                    // seller bid floor
    timestamp: Instant,                    // ad request timestamp or auction timestamp
    duration: Option[Long],                // request duration in millis
    sessionNumber: Option[Long],
    impressionNumber: Option[Long],
    sdkless: Option[Boolean],
    skadn: Option[SkAdnRequest] = None,
    predictedBidFloor: Option[Double],
    predictionInfo: Option[String],
    sessionId: Option[String],
    mediationStackId: Option[String],
    bmIfv: Option[String],
    dm: Option[String] = None,
    dmVer: Option[String] = None,
    vpt: Option[Int],
    initialPlacements: List[String],
    postbid: Option[Boolean],
    supplyParameters: Option[SupplyParameters],
    nbr: Option[String],
    consentStringVersion: Option[ConsentStringVersion],
    mediationBidRequestId: Option[String], // request `id` from root of S2S OpenRTB2 request to our exchange
    metrics: Option[Metrics],
    hva: Option[Boolean],
    appHva: Option[Boolean],
    isPurchaser: Option[Boolean],
    sourceRequestId: Option[String],       // original incoming Request ID (generated by exchange)
    originalFloor: Option[Double],         // floor which we receive from seller
    intentiqAbPercentage: Option[Int],
    intentiqAbTestUuidIsDefined: Option[Boolean],
    adRequestSchainLength: Option[Double]     // number of hops in schain (length) in ad request
  )

  object Ext {
    case class SupplyParameters(
      customTargeting: Map[String, String],
      mediationMode: Option[String],
      originalAdType: Option[String],
      mediationSdkVersion: Option[String],
      mediationSdk: Option[String]
    )
  }

  implicit val requestExtCodec: JsonValueCodec[Request.Ext] = JsonCodecMaker.make[Request.Ext](CodecMakerConfig)
  implicit val requestCodec: JsonValueCodec[Request]        = JsonCodecMaker.make[Request](CodecMakerConfig)
}
