package io.bidmachine.schema.rtb

import com.github.plokhotnyuk.jsoniter_scala.core.JsonValueCodec
import com.github.plokhotnyuk.jsoniter_scala.macros.{CodecMakerConfig, JsonCodecMaker}
import io.bidmachine.schema.adcom.Ad
import io.bidmachine.schema.analytics.discounting.Discounting
import io.bidmachine.schema.analytics.rendering.RenderingTemplate
import io.bidmachine.schema.analytics.{BidStatus, NumericPriceLevels, PriceLevels}
import io.bidmachine.schema.annotations.money
import io.bidmachine.schema.rtb.Bid.Ext._

case class Bid(
  id: Option[String],
  bidid: Option[String], // BidResponse ID
  seat: Option[String],
  item: Option[String],
  @money price: Double,
  media: Ad,
  deal: Option[String],  // Reference to the deal.id from the bid request if this bid pertains to a PMP direct deal.
  ext: Bid.Ext
)

object Bid {
  case class Ext(
    auctionBidFloorSource: String,
    bidderBidFloorSource: String,
    status: BidStatus,
    @money clearPrice: Option[Double],
    @money sellerClearPrice: Option[Double],
    sellerPriceLevels: PriceLevels,
    sellerPriceLevelsV2: NumericPriceLevels,
    lossReason: Option[String],
    priceLevels: PriceLevels,
    priceLevelsV2: NumericPriceLevels,
    auctionSeq: Option[Int],
    advertisedAppId: Option[String],
    cached: Boolean,
    nativeAssets: Option[List[String]],
    placementId: Option[String],
    skadn: Option[SkAdnResponse] = None,
    discounting: Option[Discounting],
    @money roundedPrice: Option[Double],
    billingTrackerLength: Option[Int],
    `rp_creativeapi`: Option[Long],
    @money supplyFee: Option[Double],
    nurlResponseStatus: Option[String],
    @money netPrice: Option[Double],  // bid price after subtracting dsp fee
    @money demandFee: Option[Double], // dsp fee subtracted from bid price before auction
    obdPredictedProbability: Option[Double],
    obdFee: Option[Double],
    obdSellerClearingPrice: Option[Double],
    discrepancyProjection: Option[DiscrepancyProjection],
    sizes: Option[Sizes] = None,
    invalidBidInfo: Option[InvalidBidInfo],
    buyerId: Option[String],          // The buyer id who won the auction on the adNetwork/DSP side
    cachedLurlMillisPassed: Option[Long],
    @money cachedLurlPrice: Option[Double],
    rendering: Option[AdaptiveRenderingContext] = None,
    creativeDetectorPayload: Option[CreativeDetectorPayload] = None,
    @money buyerBidPrice: Double,     // original bid.price from Buyer bid-response
    bidBoosting: Option[BidBoosting],
    apiFramework: Option[String],
    apiFrameworkVersion: Option[String],
    bidAdt: BidAdType = BidAdType.Unknown,
    campaignType: Option[String]
  )

  object Ext {
    case class DiscrepancyProjection(
      price: Double,
      percent: Double,
      discrepancy: Option[Double],
      p2PGrossDiscrepancy: Option[Double],
      p2PFeeDiscrepancy: Option[Double]
    )

    case class InvalidBidInfo(blockedCat: Option[List[String]], blockedAdomain: Option[List[String]])

    case class Sizes(
      originalAdm: Option[Long],         // DSP ADM size
      purlTracker: Option[Long],         // DSP PURL size
      impTrackers: Option[List[Long]],   // DSP imp trackers sizes
      clickTrackers: Option[List[Long]], // DSP click trackers sizes
      lurlTrackers: Option[List[Long]],  // DSP lurl trackers sizes
      nurl: Option[Long],                // DSP bid.NURL size
      burl: Option[Long],                // DSP bid.BURL size
      lurl: Option[Long],                // DSP bid.LURL size
      context: Option[Long] = None,      // Exchange context size
      adm: Option[Long] = None,          // Exchange ADM size
      bidResponse: Option[Long] = None   // Exchange Bid Response size
    )

    case class AdaptiveRenderingContext(
      phase: Option[Int],
      element: Option[String],
      autoclick: Option[Boolean],
      duplicated: Option[Boolean],
      template: Option[RenderingTemplate]
    )

    case class CreativeDetectorPayload(
      phase: Int,
      result: Int,
      component: String
    )

    /**
     * [[https://appodeal.atlassian.net/browse/APDX-13553 | APDX-13553]] In this case, when boosting is applied,
     * [[io.bidmachine.schema.rtb.Bid#price]] is evaluated as [[io.bidmachine.schema.rtb.Bid.Ext#buyerBidPrice]] +
     * [[io.bidmachine.schema.rtb.Bid.Ext.BidBoosting#amount]]
     */
    case class BidBoosting(
      @money amount: Double,    // additive value to bid price (delta), evaluated as `bid.ext.buyerBidPrice * BidBoosting.factor`
      factor: Double,           // multiplier by which we increase the original bid price (bid.ext.buyerBidPrice)
      noBudget: Option[Boolean] // flag at which the budget constraint is ignored
    )
  }

  implicit val bidExtCodec: JsonValueCodec[Bid.Ext]                                    = JsonCodecMaker.make[Bid.Ext](CodecMakerConfig)
  implicit val bidCodec: JsonValueCodec[Bid]                                           = JsonCodecMaker.make[Bid](CodecMakerConfig)
  implicit val adaptiveRenderingContextCodec: JsonValueCodec[AdaptiveRenderingContext] =
    JsonCodecMaker.make[AdaptiveRenderingContext](CodecMakerConfig)
}
