package io.bidmachine.rollouts

import cats.Eq
import cats.syntax.either._
import eu.timepit.refined.api.{RefType, Refined}
import eu.timepit.refined.numeric.Interval
import eu.timepit.refined.string.MatchesRegex
import eu.timepit.refined.types.string.NonEmptyString
import eu.timepit.refined.{W, refineV}
import io.bidmachine.rollouts.targeting.validation.{AttrInfo, AttrInfoProvider}
import io.circe.refined._
import io.circe.{Decoder, Encoder}
import io.estatico.newtype.macros.newtype

package object model {

  @newtype case class AttributeId(value: EntityIdWithUnderscore)
  object AttributeId {
    def from(s: String): Option[AttributeId] =
      RefType.applyRef[EntityIdWithUnderscore](s).toOption.map(AttributeId(_))
  }

  @newtype case class AllocationLabel(value: EntityId)

  @newtype case class EnvironmentId(value: EntityId)

  object EnvironmentId {
    implicit val envIdEq: Eq[EnvironmentId] = Eq.fromUniversalEquals

    def from(s: String): Either[String, EnvironmentId] =
      EntityId.from(s).bimap(e => s"Expected environment identifier, but got [$s]. $e", EnvironmentId(_))

  }

  @newtype case class EnvironmentName(value: NonEmptyString)

  object EnvironmentName {
    def from(s: String): Either[String, EnvironmentName] =
      NonEmptyString.from(s).bimap(e => s"Expected valid environment name, but got [$s]. $e", EnvironmentName(_))

  }

  @newtype case class FeatureId(value: EntityId)
  object FeatureId {
    implicit val featureIdEq: Eq[FeatureId] = Eq.fromUniversalEquals

    def from(s: String): Either[String, FeatureId] =
      EntityId.from(s).bimap(e => s"Expected feature identifier, but got [$s]. $e", FeatureId(_))
  }

  @newtype case class FeatureName(value: NonEmptyString)
  @newtype case class FeatureDescription(value: String)

  @newtype case class VariableId(value: String)

  @newtype case class NamespaceId(value: EntityId)

  object NamespaceId {
    def safeApply(id: String): Either[String, NamespaceId] =
      RefType.applyRef[EntityId](id).map(NamespaceId(_))
  }
  @newtype case class NamespaceName(value: NonEmptyString)

  object NamespaceName {
    def from(str: String): Either[String, NamespaceName] =
      NonEmptyString.from(str).map(NamespaceName(_))
  }

  @newtype case class ExperimentId(value: EntityId)

  object ExperimentId {
    implicit val expIdEq: Eq[ExperimentId] = Eq.fromUniversalEquals
  }

  @newtype case class ExperimentName(value: NonEmptyString)

  @newtype case class VariantId(value: EntityId)

  object VariantId {
    implicit val variantIdEq: Eq[VariantId] = Eq.fromUniversalEquals
  }

  @newtype case class VariantName(value: NonEmptyString)

  @newtype case class TagName(value: EntityIdWithUnderscore)

  @newtype case class ScopeId(value: EntityId)

  object ScopeId {
    implicit val scopeIdEq: Eq[ScopeId] = Eq.fromUniversalEquals
  }

  @newtype case class ScopeName(value: EntityIdWithUnderscore)

  type EntityId               = String Refined MatchesRegex["^([a-z0-9]+-)*([a-z0-9]+)$"]
  object EntityId {
    def from(s: String): Either[String, EntityId] =
      RefType.applyRef[EntityId](s)
  }
  type EntityIdWithUnderscore = String Refined MatchesRegex["^([a-z0-9]+[-_])*([a-z0-9]+)$"]

  type PercentRefinement = Interval.Closed[W.`0.0`.T, W.`100.0`.T]

  @newtype case class GroupId(value: EntityId)

  type Versioned[A] = (A, Long)

  @newtype case class Percent(value: Double Refined PercentRefinement) {
    def double = this.value.value
  }

  object Percent {
    implicit val percentDecoder: Decoder[Percent] = deriving
    implicit val percentEncoder: Encoder[Percent] = deriving

    def safeApply(p: Double): Option[Percent] =
      refineV[PercentRefinement](p).toOption.map(Percent(_))

    def unsafe(p: Double): Percent = Percent(
      refineV[PercentRefinement](p).getOrElse(throw new RuntimeException(s"invalid percent $p"))
    )
  }

  def attrInfoProvider(attributes: List[Attribute]): AttrInfoProvider =
    attributes.zipWithIndex.map { case (a, index) =>
      (a.attr, AttrInfo(a.tpe, a.allowedValues.getOrElse(Set.empty), index))
    }.toMap.get(_)
}
